/******************************************************************/
/*																						*/
/*	includes																			*/
/*																						*/
/******************************************************************/

#include <stdio.h>

#include <iapl.h>
#include <font.h>
#include <graphics.h>

#include <iaplgraf.h>

/******************************************************************/
/*																						*/
/* defines																			*/
/*																						*/
/******************************************************************/

#define PARAMS 10
#define TILEMAX 64
#define WHEELMAX 720
#define OMEGA 32767

#define GET(Z) \
	(where [Z].col < 0 ? where [Z].value : gdata (row, where [Z].col))

#define X 0
#define Y 1
#define C 2
#define F 3
#define XS 4
#define YS 5
#define XI 6
#define YI 7
#define L 8
#define I 9

#define ABS(X) ((X) < 0 ? -(X) : (X))
#define SIGN(X) ((X) < 0 ? -1 : (X) > 0)
#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))

/******************************************************************/
/*																						*/
/* globals																			*/
/*																						*/
/******************************************************************/

static int far *data;
static int rows, cols, esize;
static int defaults [PARAMS];
static int limit [PARAMS];
static int wheelmax;
static int gxs, gys;
static int gc [4];
static int gj, gt;
static int gscale;
static int gxscale;
static int gyscale;

static byte tile [TILEMAX] [TILEMAX] = {{0}};
static struct {int x, y;} tilesize = {1, 1};

static byte wheel [WHEELMAX] = {0};
static int wheelsize = 1;

static int (*plot_routine) (int, int, int, int);
static int (*get_routine) (int, int);

/******************************************************************/
/*																						*/
/*	set color																		*/
/*																						*/
/******************************************************************/

void gcolor (c, i) register int c, i;
{
	static int color [4] = {0, 0, 0, -1};

	color [1] = ~c;
	color [2] = c;

	gc [0] = color [i & 3];
	gc [1] = color [(i >>= 2) & 3];
	gc [2] = color [(i >>= 2) & 3];
	gc [3] = color [(i >>= 2) & 3];

	gj = (i >>= 2) & 1;
	gt = (i >>= 1) & 1;
}

/******************************************************************/
/*																						*/
/*	plot pixel																		*/
/*																						*/
/******************************************************************/

int gplot (xo, yo, xp, yp, c) int xo, yo; int xp, yp; int c;
{
	static char count;

	int xe, ye, f, t;
	register int x = xp;
	register int y = yp;
	int d =
		!gj ? c :
		!gt ?	(int) tile [x % tilesize.x] [y % tilesize.y] :
			(int) tile [(x - xo) % tilesize.x] [(y - yo) % tilesize.y];

	if ((unsigned) x < (unsigned) limit [X] &&
		(unsigned) y < (unsigned) limit [Y])
			if (gscale)
			{
				xe = x + 1 << gxscale;
				ye = y + 1 << gyscale;
				f = gc [0] & ~d | gc [2] & d;
				t = gc [1] & ~d | gc [3] & d;

				for (x = xp << gxscale; x < xe; x++)
					for (y = yp << gyscale; y < ye; y++)
						(*plot_routine) (x, y, f, t);

			}
			else (*plot_routine) (x, y,
				gc [0] & ~d | gc [2] & d, gc [1] & ~d | gc [3] & d);

	return count++ || !aplpending (1);
}

/******************************************************************/
/*																						*/
/*	get data item																	*/
/*																						*/
/******************************************************************/

int gdata (row, col) int row, col;
{
	register int offset = cols * row + col;

	return esize == 2 ? data [offset] :
		esize == 1 ? (int) ((unsigned char far *) data) [offset] :
		(int) (((unsigned char far *) data) [offset >> 3] >> (offset & 7)) & 1;
}

/******************************************************************/
/*																						*/
/*	plot point																		*/
/*																						*/
/******************************************************************/

int gpoint (x, y) register int x, y;
{
	static int wi = 0;
	int c = (int) wheel [wi = (wi+1) % wheelsize];
	int xo, yo, xe, ye;

	if (gxs || gys)
	{
		xo = x;
		yo = y;
		xe = x + gxs + 1;
		ye = y + gys + 1;

		for (x = xo; x < xe; x++)
			for (y = yo; y < ye; y++)
				if (!gplot (xo, yo, x, y, c)) return 0;

		return 1;
	}
	else return gplot (x, y, x, y, c);
}

/******************************************************************/
/*																						*/
/*	plot line																		*/
/*																						*/
/******************************************************************/

int gline (x, y, xe, ye) register int x, y; int xe, ye;
{
	int mx = ABS (xe - x);
	int my = ABS (ye - y);
	int sx = SIGN (xe - x);
	int sy = SIGN (ye - y);
	int delta = mx < my ? my / 2 : MIN (my - (mx + 1) / 2, 0);

	while (x != xe || y != ye)
	{
		if (!gpoint (x, y)) return 0;

		if (delta >= 0) delta -= mx, y += sy;
		if (delta < 0) delta += my, x += sx;
	}

	return gpoint (x, y);
}

/******************************************************************/
/*																						*/
/*	get graphics parameters														*/
/*																						*/
/******************************************************************/

int gparam (alpha, size) int far *alpha; int size;
{
	register int i;

	if (size != PARAMS + 2) return 0;

	for (i = 0; i < PARAMS; i++) alpha [i] = limit [i];

	if (alpha [0]) alpha [PARAMS] = TILEMAX, alpha [PARAMS + 1] = wheelmax;

	return 1;
}

/******************************************************************/
/*																						*/
/*	put graphics object															*/
/*																						*/
/******************************************************************/

int gput (alpha, size) int far *alpha; int size;
{
	struct {int col, value;} where [PARAMS];

	int i, n, flag, jn, jf, f, x, y, lastx, lasty, row, col;

	for (col = i = 0; i < size; i++)
		if (alpha [i] == OMEGA)
		{
			if (col >= cols) return 0;
			where [i].col = col++;
		}
		else
		{
			where [i].col = -1;
			where [i].value = alpha [i];
		}

	for (i = size; i < PARAMS; i++)
	{
		if (col < cols) where [i].col = col++;
		else where [i].col = -1, where [i].value = defaults [i];
	}

	for (flag = 1, n = f = row = 0; flag && row < rows; row++)
	{
		gxs = GET (XS);
		gys = GET (YS);
		gcolor (GET (C), GET (F));

		if (!n--) n = GET (L), jn = 0;
		if (!f--) f = GET (I), jf = 1;

		lastx = x;
		lasty = y;

		x = GET (X) + jf * GET (XI);
		y = GET (Y) + jf++ * GET (YI);

		flag = jn++ ? gline (lastx, lasty, x, y) : n ? 1 : gpoint (x, y);
	}

	return 1;
}

/******************************************************************/
/*																						*/
/*	set tile																			*/
/*																						*/
/******************************************************************/

int gtile (alpha, size) int far *alpha; int size;
{
	register int row, col;

	if (!rows || !cols || rows > TILEMAX || cols > TILEMAX) return 0;

	for (row = 0; row < rows; row++)
		for (col = 0; col < cols; col++)
			tile [col] [rows - row - 1] = (byte) gdata (row, col);

	tilesize.x = cols;
	tilesize.y = rows;

	return 1;
}

/******************************************************************/
/*																						*/
/*	set wheel																		*/
/*																						*/
/******************************************************************/

int gwheel (alpha, size) int far *alpha; int size;
{
	register int col;

	if (rows != 1 || !cols || cols > wheelmax) return 0;

	for (col = 0; col < cols; col++) wheel [col] = (byte) gdata (0, col);

	wheelsize = cols;

	return 1;
}

/******************************************************************/
/*																						*/
/* draw raster																		*/
/*																						*/
/******************************************************************/

int graster (alpha, size) int far *alpha; int size;
{
	int value [PARAMS];
	int row, col;
	int i;
	int xi, yi;
	register int x, y;

	for (i = 0; i < PARAMS; i++)
		value [i] = i < size ? alpha [i] : defaults [i];

	gxs = value [XS];
	gys = value [YS];
	xi = value [XI];
	yi = value [YI];

	for (row = rows - 1, y = value [Y]; row >= 0; row--, y += yi + gys + 1)
		for (col = 0, x = value [X]; col < cols; col++, x += xi + gxs + 1)
		{
			gcolor (gdata (row, col), value [F]);
			if (!gpoint (x, y)) return 0;
		}

	return 1;
}

/******************************************************************/
/*																						*/
/* draw characters																*/
/*																						*/
/******************************************************************/

int gtext (alpha, size) int far *alpha; int size;
{
	int value [PARAMS];
	int row, col;
	int i, j, n;
	int x, y, xi, yi, xq, yq, xj, yj;
	byte *ptr;
	register int xp, yp;

	for (i = 0; i < PARAMS; i++)
		value [i] = i < size ? alpha [i] : defaults [i];

	gxs = value [XS];
	gys = value [YS];
	gcolor (value [C], value [F]);
	xj = value [XI] + (gxs + 1 << 3);
	yj = value [YI] + (gys + 1 << 3);
	xi = value [I] ? 0 : value [XI];
	yi = value [I] ? 0 : value [YI];

	for (row = rows, x = value [X], y = value [Y]; row--; x += xi, y += yj)
	{
		for (col = 0, xq = x, yq = y; col < cols; col++, xq += xj, yq += yi)
		{
			for (i = 7, yp = yq, ptr = cga_font [gdata (row, col) & 0xFF] + 6; i;
				i--, ptr--, yp += gys + 1)
			{
				for (j = 7, xp = xq, n = (int) *ptr; j;
					j--, xp += gxs + 1, n <<= 1)
				{
					if (n & 0x40) if (!gpoint (xp, yp)) return 0;
				}
			}
		}
	}

	return 1;
}

/******************************************************************/
/*																						*/
/* get raster																		*/
/*																						*/
/******************************************************************/

int gget (alpha, size) int far *alpha; int size;
{
	int xe = alpha [X] + cols;
	int ye = alpha [Y] + rows;
	int far *ptr = data;
	register int x, y;

	for (y = ye - 1; y >= alpha [Y]; y--)
		for (x = alpha [X]; x < xe; x++)
			*ptr++ = (unsigned) x < (unsigned) limit [X] &&
				(unsigned) y < (unsigned) limit [Y] ?
					(*get_routine) (x << gxscale, y << gyscale) : 0;

	return 1;
}

/******************************************************************/
/*																						*/
/* top level graphics routine													*/
/*																						*/
/******************************************************************/

int graphics (alpha, omega) ARRAY *alpha, *omega;
{
	static int (*fns []) (int far *, int) = 
	{
		gparam, gput, gtile, gwheel, graster, gtext, gget
	};

	int code = *alpha->idata;

	esize = (int) *omega->esize;

	if (code < 100 || code >= 100 + sizeof (fns) / sizeof (*fns) ||
		esize == 6 || *omega->rank > 2 ||
		code != 105 && esize == 1 ||
		code == 106 &&
			(esize !=2 || *omega->rank != 2 ||
			*alpha->nels != 3 && *alpha->nels != 5) ||
		code != 100 && *alpha->nels - 1 > PARAMS)
			return 0;

	data = omega->idata;

	if (!*omega->rank) rows = cols = 1;
	else if (*omega->rank == 1) rows = 1, cols = omega->shape [0];
	else rows = omega->shape [0], cols = omega->shape [1];

	return (*fns [*alpha->idata - 100]) (alpha->idata + 1, *alpha->nels - 1);
}

/******************************************************************/
/*																						*/
/* initialise graphics															*/
/*																						*/
/******************************************************************/

void graphics_init (display_code, scale_code) char display_code, scale_code;
{
	static struct PAR {char code; int x, y, c} params [] =
	{
		{'H', 720, 348,  2},
		{'C', 640, 200,  2},
		{'M', 320, 200,  4},
		{'N', 640, 350, 16},
		{  0,   0,   0,  0}
	};

	struct PAR *ptr;
	register int i;

	for (ptr = params; ptr->code; ptr++) if (ptr->code == display_code) break;

	plot_routine =
		display_code == 'H' ? hercules_plot :
		display_code == 'C' ? cga_plot :
		display_code == 'M' ? mga_plot :
		display_code == 'N' ? ega_plot : 0;
	get_routine =
		display_code == 'H' ? hercules_get :
		display_code == 'C' ? cga_get :
		display_code == 'M' ? mga_get :
		display_code == 'N' ? ega_get : 0;

	gscale = (int) (scale_code - '0');
	gxscale = gscale + 1 >> 1;
	gyscale = gscale >> 1;

	if (ptr->x)
	{
		limit [X] = limit [XS] = limit [XI] = ptr->x >> gxscale;
		limit [Y] = limit [YS] = limit [YI] = ptr->y >> gyscale;
		limit [C] = ptr->c;
		limit [F] = 1024;
		limit [I] = limit [L] = 32767;
	}
	else for (i = 0; i < PARAMS; i++) limit [i] = 0;

	for (i = 0; i < PARAMS; i++) defaults [i] = 0;
	defaults [F] = -1;

	wheel [0] = tile [0] [0] = -1;

	wheelmax = MAX (limit [X], limit [Y]);
}
}
], limit [Y]);
}
limit [Y]);
}
